/* -*- Mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella <vasta@ragnu.it>
 */
 
 /**
	*\file lupin-buf.h
	*\author Marco Guastella (contact:vasta@ragnu.it)
	*\brief Header file to define the buffer functions
*/

#ifndef	_LIBLUPINBUF_H
#define	_LIBLUPINBUF_H	1

/**
	\defgroup buf buf section
	
*/

/**
	*\ingroup buf 
	*\brief  buffer  type
*/
typedef struct buffer
{
	/**
	*\ingroup buf 
	*\brief  pointer to memory buffer
	*/
	void *mem;
	
	/**
	*\ingroup buf 
	*\brief  buffer size
	*/
	l_size size;
	
	/**
	*\ingroup buf 
	*\brief  allocated memory mem
	*/
	l_size msize;
}*l_buf;

/**
	\ingroup buf
	\return a new buffer
	\brief creates a new buffer
*/
l_buf l_newbuf();

/**
	\ingroup buf
	\param[in] pbuf pointer to the buffer to eliminate
	\brief eliminates the buffer pointed to by pbuf and free the memory allocated to it
*/
void l_freebuf(l_buf *pbuf);

/**
	\ingroup buf
	\param[in] buf input buffer
	\return size input buffer
	\brief returns the size of the input buffer
*/
l_size l_sizebuf(l_buf buf);

/**
	\ingroup buf
	\param[in] buf input buffer
	\brief reset the input buffer and initializes the file size zero
*/
void l_resetbuf(l_buf buf);

/**
	\ingroup buf
	\param[out] buf buffer to change size
	\param[in] ns new size buf
	\brief change the buffer size if ns < size buf reset memory
*/
l_size l_newsizebuf(l_buf buf,l_size ns);

/**
	\ingroup buf
	\param[in] buf input buffer
	\param[in] nbyte the position of bytes returned
	\return byte position nbyte
	\brief returns the value of the byte position of the input buffer nbyte
*/
l_byte l_bytebuf(l_buf buf,l_num nbyte);

/**
	\ingroup buf
	\param[in] buf input buffer
	\param[in] a initial point from where to draw the bit
	\param[in] nbits number of bits to charge(MAX 8)
	\return the byte format with the bits taken
	\brief  download the buffer buf nbits starting from point a and returns the byte format from these bits
*/
l_byte l_bitseqbuf(l_buf buf,l_num a,l_num nbits);

/**
	\ingroup buf
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief writes the memory mem of size len into the buffer buf starting at the position point
*/
l_size l_writebuf(l_buf buf,l_num point,void *mem,l_size len);

/**
	\ingroup buf
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief reset buffer and writes the memory mem of size len into the buffer buf starting at the position point
*/
l_size l_nwritebuf(l_buf buf,l_num point,void *mem,l_size len);

/**
	\ingroup buf
	\param[out] buf write buffer
	\param[in] mem memory to be written
	\param[in] len length of the message to be written
	\return the new buffer size
	\brief  concatenates the memory mem of size len into the buffer buf
*/
l_size l_cwritebuf(l_buf buf,void *mem,l_size len);

/**
	\ingroup buf
	\param[out] buf write buffer
	\param[in] point point buffer to write
	\param[in] fmt fmt format string
	\param[in] ... arguments
	\return the new buffer size
	\brief produces an output based on the description of the 
	* format string and writes it to buffer buf.
	*The format string has the format "v [: v]" where the values of v are:
	*b = l_byte;
	*w = l_word;
	*dw= l_dword;
	*s=string
*/
l_size l_fwritebuf(l_buf buf,l_num point,const l_str fmt,...);

/**
	\ingroup buf
	\param[in] buf read buffer
	\param[in] point point buffer to read
	\param[in] fmt fmt format string
	\param[out] ... arguments
	\return the number byte read
	\brief produces an input based on the description of the 
	* format string and reads the values from the buffer buf and copies them in the arguments. 
	*Attention arguments must be pointers.
	*The format string has the format "v [: v]" where the values of v are:
	*b = l_byte;
	*w = l_word;
	*dw= l_dword;
	*s.n= string of length n
*/
l_size l_freadbuf(l_buf buf,l_size point,const l_str fmt,...);


/**
	\ingroup buf
	\param[out] dbuf destination buffer
	\param[in]   sbuf source buffer
	\return  the new destination buffer size
	\brief copy the buffer sbuf into the buffer dbuf 
*/	
l_size l_copybuf(l_buf dbuf,l_buf sbuf);

/**
	\ingroup buf
	\param[out] dbuf destination buffer
	\param[in]   sbuf source buffer
	\return  the new destination buffer size
	\brief concatenates the buffer sbuf into the buffer dbuf 
*/	
l_size l_catbuf(l_buf dbuf,l_buf sbuf);

/**
	\ingroup buf
	\param[out] dbuf destination buffer
	\param[in]   pd point buffer sbuf to write
	\param[in]   sbuf source buffer
	\param[in]   ps  point buffer sbuf to copy
	\param[in]   len number of bytes to copy
	\return  the new destination buffer size
	\brief copy len bytes from the buffer sbuf ps from the point buffer at the point dbuf pd
*/
void l_copymembuf(l_buf dbuf,l_num pd,l_buf sbuf,l_num ps,l_size len);

/**
	\ingroup buf
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard output (type string)
*/
void l_outbuf(l_buf buf);

/**
	\ingroup buf
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard error (type string)
*/
void l_errbuf(l_buf buf);

/**
	\ingroup buf
	\param[in] buf print buffer
	\brief print the contents of the buffer to standard output (sequence of hexadecimal characters)
*/
void l_outexabuf(l_buf buf);

#endif /*end lupin-buf.h*/
